﻿using System.IO.Ports;
using System;
using System.Windows.Forms;

namespace WindowsFormsApplication1
{
    public class Bld04aCommand
    {
        // The command to communicate with the BLD04A.
        private const byte CMD_HEADER               = 0x00;
        private const byte CMD_TEST_COMM            = 0xaa;
        private const byte CMD_SET_BAUD_RATE        = 0x22;

        private const byte CMD_STOP                 = 0xf0;
        private const byte CMD_BRAKE                = 0xf1;
        private const byte CMD_RUN                  = 0xf2;
        private const byte CMD_DIRECTION_0          = 0xf3;
        private const byte CMD_DIRECTION_1          = 0xf4;
        private const byte CMD_RESET_ALM            = 0xf5;
        private const byte CMD_SET_SPEED            = 0xf6;
        private const byte CMD_SET_PARALLEL         = 0xf7;
        private const byte CMD_READ_SPEED           = 0xf8;

        // The SerialPort object.
        private SerialPort serialPort;

        // The read and write buffer for the serial port.
        private byte[] serialPortWriteBuffer = new byte[10];
        private byte[] serialPortReadBuffer = new byte[10];



        public Bld04aCommand(SerialPort serialPort)
        {
            this.serialPort = serialPort;

            // Every command begins with 0xaa, 0xaa. So we put them at the begining of the write buffer.
            this.serialPortWriteBuffer[0] = CMD_HEADER;
            this.serialPortWriteBuffer[1] = CMD_HEADER;
        }



        // Test the communication with the BLD04A.
        public bool isBld04aConnected()
        {
            // Make sure the COM Port is already opened.
            if (this.serialPort.IsOpen == true)
            {
                // Put the command into the write buffer.
                this.serialPortWriteBuffer[2] = CMD_TEST_COMM;

                // Send the command.
                this.serialPort.Write(this.serialPortWriteBuffer, 0, 3);

                // Clear the read buffer.
                Array.Clear(this.serialPortReadBuffer, 0, this.serialPortReadBuffer.Length);

                // Read the respond.
                try
                {
                    // Read 3 bytes.
                    for (int i = 0; i < 3; i++)
                    {
                        this.serialPortReadBuffer[i] = (byte)this.serialPort.ReadByte();
                    }
                }
                catch (System.Exception)
                {
                }

                // Check the received value.
                if ((this.serialPortReadBuffer[0] == CMD_HEADER) &&
                    (this.serialPortReadBuffer[1] == CMD_HEADER) &&
                    (this.serialPortReadBuffer[2] == CMD_TEST_COMM))
                {
                    return true;
                }

            }

            return false;
        }



        // Stop the motor.
        public void stopMotor()
        {
            // Make sure the COM Port is already opened.
            if (this.serialPort.IsOpen == true)
            {
                // Put the command into the write buffer.
                this.serialPortWriteBuffer[2] = CMD_STOP;

                // Send the command.
                this.serialPort.Write(this.serialPortWriteBuffer, 0, 3);
            }
        }



        // Brake the motor.
        public void brakeMotor()
        {
            // Make sure the COM Port is already opened.
            if (this.serialPort.IsOpen == true)
            {
                // Put the command into the write buffer.
                this.serialPortWriteBuffer[2] = CMD_BRAKE;

                // Send the command.
                this.serialPort.Write(this.serialPortWriteBuffer, 0, 3);
            }
        }



        // Run the motor.
        public void runMotor()
        {
            // Make sure the COM Port is already opened.
            if (this.serialPort.IsOpen == true)
            {
                // Put the command into the write buffer.
                this.serialPortWriteBuffer[2] = CMD_RUN;

                // Send the command.
                this.serialPort.Write(this.serialPortWriteBuffer, 0, 3);
            }
        }



        // Set the motor direction.
        public void setMotorDirection(byte direction)
        {
            // Make sure the COM Port is already opened.
            if (this.serialPort.IsOpen == true)
            {
                // Make sure the direction is either 0 or 1;
                if (direction == 0 || direction == 1)
                {
                    // Put the command into the write buffer.
                    if (direction == 0)
                    {
                        this.serialPortWriteBuffer[2] = CMD_DIRECTION_0;
                    }
                    else
                    {
                        this.serialPortWriteBuffer[2] = CMD_DIRECTION_1;
                    }
                    

                    // Send the command.
                    this.serialPort.Write(this.serialPortWriteBuffer, 0, 3);
                }
            }
        }



        // Reset the alarm.
        public void resetMotorAlarm()
        {
            // Make sure the COM Port is already opened.
            if (this.serialPort.IsOpen == true)
            {
                // Put the command into the write buffer.
                this.serialPortWriteBuffer[2] = CMD_RESET_ALM;

                // Send the command.
                this.serialPort.Write(this.serialPortWriteBuffer, 0, 3);
            }
        }



        // Change to parallel mode.
        public void setParallelMode()
        {
            // Make sure the COM Port is already opened.
            if (this.serialPort.IsOpen == true)
            {
                // Put the command into the write buffer.
                this.serialPortWriteBuffer[2] = CMD_SET_PARALLEL;

                // Send the command.
                this.serialPort.Write(this.serialPortWriteBuffer, 0, 3);
            }
        }



        // Set motor speed.
        public void setMotorSpeed(Int16 speed)
        {
            // Make sure the COM Port is already opened.
            if (this.serialPort.IsOpen == true)
            {
                // Put the command into the write buffer.
                this.serialPortWriteBuffer[2] = CMD_SET_SPEED;
                this.serialPortWriteBuffer[3] = (byte)(speed >> 8);
                this.serialPortWriteBuffer[4] = (byte)(speed & 0x00ff);

                // Send the command.
                this.serialPort.Write(this.serialPortWriteBuffer, 0, 5);
            }
        }



        // Read the motor speed.
        public Int16 readMotorSpeed()
        {
            // Make sure the COM Port is already opened.
            if (this.serialPort.IsOpen == true)
            {
                // Put the command into the write buffer.
                this.serialPortWriteBuffer[2] = CMD_READ_SPEED;

                // Send the command.
                this.serialPort.Write(this.serialPortWriteBuffer, 0, 3);

                // Clear the read buffer.
                Array.Clear(this.serialPortReadBuffer, 0, this.serialPortReadBuffer.Length);

                // Read the respond.
                try
                {
                    // Read 4 bytes.
                    for (int i = 0; i < 4; i++)
                    {
                        this.serialPortReadBuffer[i] = (byte)this.serialPort.ReadByte();
                    }
                }
                catch (System.Exception)
                {
                    return -1;
                }

                // Make sure the received header is correct.
                if ((this.serialPortReadBuffer[0] == CMD_HEADER) &&
                    (this.serialPortReadBuffer[1] == CMD_HEADER))
                {
                    // Get the speed and return it.
                    int speed = (this.serialPortReadBuffer[2] << 8) + this.serialPortReadBuffer[3];

                    return (Int16)speed;
                }
            }

            return -1;
        }



        // Set new baud rate.
        public void setBaudRate(Int16 value)
        {
            // Make sure the COM Port is already opened.
            if (this.serialPort.IsOpen == true)
            {
                // Put the command into the write buffer.
                this.serialPortWriteBuffer[2] = CMD_SET_BAUD_RATE;
                this.serialPortWriteBuffer[3] = (byte)(value >> 8);
                this.serialPortWriteBuffer[4] = (byte)(value & 0x00ff);

                // Send the command.
                this.serialPort.Write(this.serialPortWriteBuffer, 0, 5);
            }
        }





    }
}
